﻿#ConnetionFailures.ps1 Created by Rob Zylowski Citrix Consulting Sr. Architect.
#Version 1.0 5-27-2022

#<This software application is provided to you “as is” with no representations, warranties or conditions of any kind. You may use and distribute it at your own risk. CITRIX DISCLAIMS ALL WARRANTIES WHATSOEVER, EXPRESS, IMPLIED, WRITTEN, ORAL OR STATUTORY, INCLUDING WITHOUT LIMITATION WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NONINFRINGEMENT. Without limiting the generality of the foregoing, you acknowledge and agree that (a) the software application may exhibit errors, design flaws or other problems, possibly resulting in loss of data or damage to property; (b) it may not be possible to make the software application fully functional; and (c) Citrix may, without notice or liability to you, cease to make available the current version and/or any future versions of the software application. In no event should the code be used to support of ultra-hazardous activities, including but not limited to life support or blasting activities. NEITHER CITRIX NOR ITS AFFILIATES OR AGENTS WILL BE LIABLE, UNDER BREACH OF CONTRACT OR ANY OTHER THEORY OF LIABILITY, FOR ANY DAMAGES WHATSOEVER ARISING FROM USE OF THE SOFTWARE APPLICATION, INCLUDING WITHOUT LIMITATION DIRECT, SPECIAL, INCIDENTAL, PUNITIVE, CONSEQUENTIAL OR OTHER DAMAGES, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES. You agree to indemnify and defend Citrix against any and all claims arising from your use, modification or distribution of the code.>#

function GetBearerToken {
  param (
    [Parameter(Mandatory=$true)]
    [string] $CustomerId,
	[Parameter(Mandatory=$true)]
    [string] $ClientId,
    [Parameter(Mandatory=$true)]
    [string] $ClientSecret
  )
	$postHeaders = @{"Content-Type"="application/x-www-form-urlencoded"}
	$body = @{
		grant_type="client_credentials";
		client_id=$clientId
		client_secret=$clientSecret
	}
	$trustUrl = "https://api-us.cloud.com/cctrustoauth2/$customerId/tokens/clients "
	$response = Invoke-WebRequest -Uri $trustUrl -Method POST -Body $body -Headers $postHeaders
	$responseObj = ConvertFrom-Json $response.content
	$bearerToken = $responseObj.access_token
	return $bearerToken;
}

function Get-ScriptDirectory
{
  $Invocation = (Get-Variable MyInvocation -Scope 1).Value
  Split-Path $Invocation.MyCommand.Path
}

Function LogLine($strLine)
{
	Write-Host $strLine
	$StrTime = Get-Date -Format "MM-dd-yyyy-HH-mm-ss-tt"
	"$StrTime - $strLine " | Out-file -FilePath $LogFile -Encoding ASCII -Append
}

function Get-OData {
    [CmdletBinding()]
    param (
		[Parameter(Mandatory)]
        [double]
        $DaysToSearch
    )

	$MinStartDate = (Get-Date).AddDays(-$DaysToSearch)
    [string] $MinStartDateUTC = Get-Date -Date $MinStartDate.ToUniversalTime() -Format 'yyyy\-MM\-dd\THH\:mm\:ss'
	
	$StartDate = $MinStartDateUTC + "Z"
	#Use the ConnectionFailureLogs API call.  For other regions change the api-us.cloud.com to the appropriate url
	$root = "https://api-us.cloud.com/monitorodata/ConnectionFailureLogs()"

	#We will filter by date 
	$filter="(CreatedDate gt $StartDate)"
	#Include any fields that are wanted in the output in the select statements
	#Expand is used to connect another table
	#Here we include the Session,Machine and user tables along with the ConnectionFailureLogs from the API call
	$SelectExpand = "`$expand=Session,Machine(`$expand=DesktopGroup,Catalog),User"

	#We want to order so our last record is the latest session
	$OrderBy="`$OrderBy=FailureDate "

	if ($GLOBAL:XDAuthToken -like "CWSAuth*")
	{
		$token = $GLOBAL:XDAuthToken
	}
	else
	{
		$token = "CwsAuth Bearer="+$bearerToken
	}
	$KeepLooking = $True
	While ($KeepLooking)
	{
		if ($NextURI -eq $null)
		{
		    [Uri] $uri = "$root`?`$filter=$filter&$SelectExpand&$OrderBy" 
		}
		else
		{
			$URI = $NextURI
		}
		
		try {

			$Header = @{"Authorization"="$token";"Citrix-CustomerId"="$CustomerId"}
			Logline "URI: $uri"
			$response = Invoke-WebRequest -Uri $URI -Header $Header -ErrorAction 'Continue'
			$JSONResponse = convertfrom-json  $response.content
			$respList=$JSONResponse.Value
			#If we get more than 100 there will be more to get
			$NextURI = $JSONResponse.'@odata.nextLink' 
			if ($NextURI -eq $null){$KeepLooking = $false}
			foreach ($r in $resplist2)
			{
				#We need to convert dates to out time zone
				$FailureDate = Convert-UTCtoLocal $r.FailureDate
				$Catalog = $r.Machine.Catalog.Name
				$DeliveryGroup = $r.Machine.DesktopGroup.Name
				$Machine = $r.Machine.DNSName
				$HostEntry = $r.Machine.HostingServerName
				$MaintMode = $r.Machine.IsInMaintenanceMode
				$Registered = $r.Machine.CurrentRegistrationState
				if ($Registered -eq 0){$RegisteredText = "No"}else{$RegisteredText = "Yes"}
				
				$OSType = $r.Machine.OSType
				$PowerState = $r.Machine.CurrentPowerState
				$UserName = $r.User.UserName
				$FullName = $r.User.FullName
				$FailureEnum = $r.ConnectionFailureEnumValue
				#Lets look up the Failure Name from our hash table
				$ConnectionFailure = $FailureCodes.get_item("$FailureEnum")
				$SessionStart = Convert-UTCtoLocal $r.Session.StartDate
				$SessionEnd = Convert-UTCtoLocal $r.Session.EndDate

				$MyObject = [pscustomobject] [ordered] @{	
					Catalog 			= $Catalog 		
					DeliveryGroup		= $DeliveryGroup	
					Machine 			= $Machine 		
					Host 				= $HostEntry 			
					MaintMode 			= $MaintMode 		
					Registered 			= $RegisteredText 		
					OSType 				= $OSType 			
					PowerState 			= $PowerState 		
					UserName 			= $UserName 			
					FullName 			= $FullName 			
					ConnectionFailure	= $ConnectionFailure
					SessionStartDate	= $SessionStart
					SessionEndDate		= $SessionEnd
				}
				Export-csv -InputObject $MyObject -Path "$OutputCsvFile" -force -NoTypeInformation -Encoding ASCII -Append
			}
		}
	    catch {
			#Lets get the error, log it and return it
			$Err = $Error[0]
			Logline "Error: [$Err]"
			return $Err
		}
	}
}

function Convert-UTCtoLocal { 
	param( 
		[parameter(Mandatory=$true)] 
		[String] $Time 
	)

	$strCurrentTimeZone = (Get-WmiObject win32_timezone).StandardName 
	$TZ = [System.TimeZoneInfo]::FindSystemTimeZoneById($strCurrentTimeZone) 
	$UTCTime = ([DateTime]$Time).ToUniversalTime()
	$LocalTime = [System.TimeZoneInfo]::ConvertTimeFromUtc($UTCTime, $TZ)
	return $LocalTime
}

#=============================================================
#Script Setup Parameters
#=============================================================

#Citrix Cloud Settings

#Set the following to $true to be prompted for cloud credentials
#This requires the remote PowerShell SDK be installed
$UseInteractiveAuth = $true

#API Key Logon Credentials
#If using the Interactive logon these three settings are not required
$CustomerId = ""
$clientId = ""
$clientSecret = ""

#Number of days to search back from now
$DaysToSearch = 10

#The output CSV filename should have no extension
$OutputFilename = "ConnectionFailures"
#=============================================================

#Get the current script folder
$ScriptSource = Get-ScriptDirectory

#Lets Create a log folder if there isn't one
$LogFolder = "$ScriptSource\Logs"
If (!(Test-Path "$LogFolder"))
{
	mkdir "$LogFolder" >$null
}
if (!(Test-Path $LogFolder))
{
	Write-Host "***Could not create log folder [$LogFolder] --- Exiting Script"
	exit
}

#Lets create a log file for this batch
$DateTimeText = Get-Date -Format "MM-dd-yyyy-HH-mm-tt"
$LogFile = "$LogFolder\$OutputFilename-$DateTimeText.txt"

#Open the csv File
$FailureCodesCSV = "$ScriptSource\SessionFailureCodes.csv"

Logline "Getting Session Failure Codes from [$FailureCodesCSV]"

if (Test-Path $FailureCodesCSV)
{
	Logline "Found CSV Lookup Users Last Logon Time"
	#Get Map csv file
	$SessionFailureCodes = Import-Csv -Path $FailureCodesCSV -Encoding ASCII
	
	$FailureCodes = @{}
	foreach ($Line in $SessionFailureCodes)
	{
		$FailureCodes.Add($Line.ErrorCode,$Line.EnumValue)
	}
}
else
{
	Logline "$FailureCodesCSV not found exiting script"
}	

#Auth use interactive logon if the following flag has been set
If  ($UseInteractiveAuth)
{
	$LookForSnapin = Get-PSSnapin -Name "Citrix.Sdk.Proxy.V1"
	If ($LookForSnapin.Name -ne "Citrix.Sdk.Proxy.V1")
	{
		Add-PSSnapin -Name "Citrix.Sdk.Proxy.V1"
	}
	Get-XDAuthentication
	$bearerToken = $GLOBAL:XDAuthToken
}
else
{
	#Lets get a bearer token so we can query Citrix Cloud
	#This is good for an hour if the script needs to run longer than an hour changes would be required.
	$bearerToken = GetBearerToken $CustomerId $clientId $clientSecret
	#or you can use the DaaS SDK to save a profile and call that here.  
	#Set-XDCredentials -CustomerId $CustomerId -APIKey $clientId -SecretKey $clientSecret -StoreAs "CitrixCloud" -ProfileType CloudApi 

	#Uncomment this and use the profile name you used when
	#Saving the profile.  Remember to comment out the line above if using this method.
	#Get-XDAuthentication -ProfileName "CitrixCloud" -Verbose
}

#Define the CSV Filename
$OutputCsvFile = "$ScriptSource\$OutputFilename-$DateTimeText.csv"

#Now for every user in the csv lets get their Last Logon Time
	
Logline ""
Logline "Looking up Session Failures for [$DaysToSearch] Days"
$sessions = Get-OData $DaysToSearch 




