﻿#LastLogon.ps1 Created by Rob Zylowski Citrix Consulting Sr. Architect.
#Version 1.0 3-15-2022
#Version 1.1 4-19-2022

#<This software application is provided to you “as is” with no representations, warranties or conditions of any kind. You may use and distribute it at your own risk. CITRIX DISCLAIMS ALL WARRANTIES WHATSOEVER, EXPRESS, IMPLIED, WRITTEN, ORAL OR STATUTORY, INCLUDING WITHOUT LIMITATION WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NONINFRINGEMENT. Without limiting the generality of the foregoing, you acknowledge and agree that (a) the software application may exhibit errors, design flaws or other problems, possibly resulting in loss of data or damage to property; (b) it may not be possible to make the software application fully functional; and (c) Citrix may, without notice or liability to you, cease to make available the current version and/or any future versions of the software application. In no event should the code be used to support of ultra-hazardous activities, including but not limited to life support or blasting activities. NEITHER CITRIX NOR ITS AFFILIATES OR AGENTS WILL BE LIABLE, UNDER BREACH OF CONTRACT OR ANY OTHER THEORY OF LIABILITY, FOR ANY DAMAGES WHATSOEVER ARISING FROM USE OF THE SOFTWARE APPLICATION, INCLUDING WITHOUT LIMITATION DIRECT, SPECIAL, INCIDENTAL, PUNITIVE, CONSEQUENTIAL OR OTHER DAMAGES, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES. You agree to indemnify and defend Citrix against any and all claims arising from your use, modification or distribution of the code.>#


function GetBearerToken {
  param (
    [Parameter(Mandatory=$true)]
    [string] $CustomerId,
	[Parameter(Mandatory=$true)]
    [string] $ClientId,
    [Parameter(Mandatory=$true)]
    [string] $ClientSecret
  )
	$postHeaders = @{"Content-Type"="application/x-www-form-urlencoded"}
	$body = @{
		grant_type="client_credentials";
		client_id=$clientId
		client_secret=$clientSecret
	}
	$trustUrl = "https://api-us.cloud.com/cctrustoauth2/$customerId/tokens/clients "
	$response = Invoke-WebRequest -Uri $trustUrl -Method POST -Body $body -Headers $postHeaders
	$responseObj = ConvertFrom-Json $response.content
	$bearerToken = $responseObj.access_token
	return $bearerToken;
}

function Get-ScriptDirectory
{
  $Invocation = (Get-Variable MyInvocation -Scope 1).Value
  Split-Path $Invocation.MyCommand.Path
}

Function LogLine($strLine)
{
	Write-Host $strLine
	$StrTime = Get-Date -Format "MM-dd-yyyy-HH-mm-ss-tt"
	"$StrTime - $strLine " | Out-file -FilePath $LogFile -Encoding ASCII -Append
}

function Get-Odata {
    [CmdletBinding()]
    param (
		[Parameter(Mandatory)]
        [double]
        $DaysToSearch,
    
		[Parameter(Mandatory)]
        [String]
        $UserName
    )

	$MinStartDate = (Get-Date).AddDays(-$DaysToSearch)
    [string] $MinStartDateUTC = Get-Date -Date $MinStartDate.ToUniversalTime() -Format 'yyyy\-MM\-dd\THH\:mm\:ss'
	
	$StartDate = $MinStartDateUTC + "Z"
	
	#We will filter by date and username 
	$filter="(CreatedDate gt $StartDate) and (User/Username eq `'$UserName`')"
	#Include any fields that are wanted in the output in the select statements
	#expand is used to connect another table
	#Here we include the user table and Connections table along with the Sessions from the API call
	$SelectExpand = "`$expand=User(`$select=username,upn),Connections(`$select=LogonStartDate,ClientName,ConnectedViaIPAddress;`$OrderBy=LogonStartDate Desc;`$top=1)"
	#We want to order so our last record is the latest session
	$OrderBy="`$OrderBy=StartDate Desc"
	

   try {
		#Use the sessions API call.  For other regions change the api-us.cloud.com to the appropriate url
		$root = "https://api-us.cloud.com/monitorodata/Sessions()"
		#Then below we take only the top=1
	    [Uri] $uri = "$root`?`$filter=$filter&$SelectExpand&$OrderBy&`$top=1" 
        Logline "URI: $uri"
		if ($GLOBAL:XDAuthToken -like "CWSAuth*")
		{
			$token = $GLOBAL:XDAuthToken
		}
		else
		{
			$token = "CwsAuth Bearer="+$bearerToken
		}
		$Header = @{"Authorization"="$token";"Citrix-CustomerId"="$CustomerId"}
        $response = Invoke-WebRequest -Uri $uri -Header $Header -ErrorAction 'Continue'
    	
		# extract what we need to return from the response
   		$JSONResponse = convertfrom-json  $response.content
		$r=$JSONResponse.Value 
		if ($r.StartDate -eq $null)
		{
		[pscustomobject] [ordered] @{
            UserName             	= $UserName
            UPN                  	= "None"
			ClientName				= "None"
			ConnectedViaIPAddress	= "None"
            CreatedDate          	= "None"
            SessionStartDate        = "None"
            SessionEndDate          = "None"
            SessionDurationMin      = "None"
			}
		}
		else
		{
			#We need to convert dates to out time zone
			$SessionStartDate = Convert-UTCtoLocal $r.StartDate
			$CeatedDate = Convert-UTCtoLocal $r.CreatedDate
			$ClientName=$r.Connections.ClientName
			$ConnectedViaIPAddress=$r.Connections.ConnectedViaIPAddress
			
			#If we have an end Date
			if (!([string]::IsNullorEmpty($r.EndDate)))
			{
				$SessionEndDate = Convert-UTCtoLocal $r.EndDate
				$SessionDuration = $SessionEndDate - $SessionStartDate 
				$SessionDurationMin = $SessionDuration.TotalMinutes.ToString("#")
			}
			else
			{
				$SessionEndDate = "None"
				$SessionDurationMin = "None"
			}
        	[pscustomobject] [ordered] @{
            UserName             	= $r.User.UserName
            UPN                  	= $r.User.Upn
			ClientName				= $ClientName
			ConnectedViaIPAddress	= $ConnectedViaIPAddress
            CreatedDate          	= $CeatedDate
            SessionStartDate        = $SessionStartDate
            SessionEndDate          = $SessionEndDate
            SessionDurationMin      = $SessionDurationMin
        	}
		}
    }
    catch {
		#Lets get the error, log it and return it
		$Err = $Error[0]
		Logline "Error: [$Err]"
		return $Err
	}
	
}

function Convert-UTCtoLocal { 
	param( 
		[parameter(Mandatory=$true)] 
		[String] $Time 
	)

	$strCurrentTimeZone = (Get-WmiObject win32_timezone).StandardName 
	$TZ = [System.TimeZoneInfo]::FindSystemTimeZoneById($strCurrentTimeZone) 
	$UTCTime = ([DateTime]$Time).ToUniversalTime()
	$LocalTime = [System.TimeZoneInfo]::ConvertTimeFromUtc($UTCTime, $TZ)
	return $LocalTime
}

#=============================================================
#Script Setup Parameters
#=============================================================

#Citrix Cloud Settings

#Set the following to $true to be prompted for cloud credentials
#This requires the remote PowerShell SDK be installed
$UseInteractiveAuth = $true

#API Key Logon Credentials
#If using the Interactive logon these three settings are not required
$CustomerId = ""
$clientId = ""
$clientSecret = ""


#Number of days to search back from now
$DaysToSearch = 90

#The csv file should be in the directory with the script
$InputCSVFilename = "UserInput.csv"

#The output CSV filename should have no extension
$OutputFilename = "LastSessionLogonTimes"
#=============================================================

#Get the current script folder
$ScriptSource = Get-ScriptDirectory

#Lets Create a log folder if there isn't one
$LogFolder = "$ScriptSource\Logs"
If (!(Test-Path "$LogFolder"))
{
	mkdir "$LogFolder" >$null
}
if (!(Test-Path $LogFolder))
{
	Write-Host "***Could not create log folder [$LogFolder] --- Exiting Script"
	exit
}

#Lets create a log file for this batch
$DateTimeText = Get-Date -Format "MM-dd-yyyy-HH-mm-tt"
$LogFile = "$LogFolder\$OutputFilename-$DateTimeText.txt"

#Open the csv File
$InputCsvFile = "$ScriptSource\$InputCSVFilename"

Logline "Getting UserNames from [$InputCsvFile]"

if (Test-Path $InputCsvFile)
{
	Logline "Found CSV Lookup Users Last Logon Time"
	#Get Map csv file
	$LastLogonUsers = Import-Csv -Path $InputCsvFile -Encoding ASCII
}
else
{
	Logline "$CsvFile not found exiting script"
	exit
}	

#Auth use interactive logon if the following flag has been set
If  ($UseInteractiveAuth)
{
	$LookForSnapin = Get-PSSnapin -Name "Citrix.Sdk.Proxy.V1"
	If ($LookForSnapin.Name -ne "Citrix.Sdk.Proxy.V1")
	{
		Add-PSSnapin -Name "Citrix.Sdk.Proxy.V1"
	}
	Get-XDAuthentication
	$bearerToken = $GLOBAL:XDAuthToken
	
	#Get CustomerID from Environment
	$Proxy = $GLOBAL:XDSDKProxy.split(".")
	$CustomerId = $Proxy[0]

}
else
{
	#Lets get a bearer token so we can query Citrix Cloud
	#This is good for an hour if the script needs to run longer than an hour changes would be required.
	$bearerToken = GetBearerToken $CustomerId $ClientId $ClientSecret
	#or you can use the DaaS SDK to save a profile and call that here.  
	#Set-XDCredentials -CustomerId $CustomerId -APIKey $clientId -SecretKey $clientSecret -StoreAs "CitrixCloud" -ProfileType CloudApi 

	#Uncomment this and use the profile name you used when
	#Saving the profile.  Remember to comment out the line above if using this method.
	#Get-XDAuthentication -ProfileName "CitrixCloud" -Verbose
}

#Define the CSV Filename
$OutputCsvFile = "$ScriptSource\$OutputFilename-$DateTimeText.csv"

#Now for every user in the csv lets get their Last Logon Time
foreach ($Line in $LastLogonUsers)
{
	
	$UserName=$Line.UserName
	#Skip if username is blank
	if ([string]::IsNullOrEmpty($UserName)){continue}
	Logline ""
	Logline "Looking up Sessions for user [$UserName] for [$DaysToSearch] Days"
	#$sessions = ""
	$sessions = Get-Odata $DaysToSearch "$UserName"
	Export-csv -InputObject $sessions -Path "$OutputCsvFile" -Append -force -NoTypeInformation
}




