
#Script ExpandUL.ps1
#Developed by Rob Zylowski Solution Delivery Architect Citrix Consulting Services.
#Version 1 9-10-2020
#Version 3.3.1 3-10-2021 - fixed bug where the comparison for if the new layer size was bigger than the old used a text comparison rather than numeric.
#                          Modified the code to force those values to integers so that the comparison woudl be numeric.
#Version 4.0 7-28-2021 - Updated for spaces in the layer path. Includes encoding the layer file path from the UI as base64.

<# *****************************************   LEGAL DISCLAIMER   *****************************************
This software / sample code is provided to you AS IS with no representations, warranties or conditions of any kind. 
You may use, modify and distribute it at your own risk. CITRIX DISCLAIMS ALL WARRANTIES WHATSOEVER, EXPRESS, IMPLIED, 
WRITTEN, ORAL OR STATUTORY, INCLUDING WITHOUT LIMITATION WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, 
TITLE AND NONINFRINGEMENT. Without limiting the generality of the foregoing, you acknowledge and agree that (a) the 
software / sample code may exhibit errors, design flaws or other problems, possibly resulting in loss of data or damage to 
property; (b) it may not be possible to make the software / sample code fully functional; and (c) Citrix may, without notice 
or liability to you, cease to make available the current version and/or any future versions of the software / sample code. 
In no event should the software / code be used to support of ultra-hazardous activities, including but not limited to life 
support or blasting activities. NEITHER CITRIX NOR ITS AFFILIATES OR AGENTS WILL BE LIABLE, UNDER BREACH OF CONTRACT OR ANY 
OTHER THEORY OF LIABILITY, FOR ANY DAMAGES WHATSOEVER ARISING FROM USE OF THE SOFTWARE / SAMPLE CODE, INCLUDING WITHOUT 
LIMITATION DIRECT, SPECIAL, INCIDENTAL, PUNITIVE, CONSEQUENTIAL OR OTHER DAMAGES, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH 
DAMAGES. Although the copyright in the software / code belongs to Citrix, any distribution of the code should include only 
your own standard copyright attribution, and not that of Citrix. You agree to indemnify and defend Citrix against any and all
claims arising from your use, modification or distribution of the code.
********************************************************************************************************** #>

Param($NewSize,$EncodedPathToVHD,$logFileName)

function Get-ScriptDirectory
{
  $Invocation = (Get-Variable MyInvocation -Scope 1).Value
  Split-Path $Invocation.MyCommand.Path
}

Function LogLine($strLine)
{
	Write-Host $strLine
	$StrTime = Get-Date -Format "MM-dd-yyyy-HH-mm-ss-tt"
	"$StrTime - $strLine " | Out-file -FilePath $LogFile -Encoding ASCII -Append
}

#We are using a base64 string to pass the vhd filename to handle spaces in the path.
#This will convert it back for us.
$PathToVHD=[System.Text.Encoding]::Unicode.GetString([System.Convert]::FromBase64String($EncodedPathToVHD))

$ScriptSource = Get-ScriptDirectory

$ProcessStartTime = Get-Date

#Create a log folder and file
$LogFolder = "$ScriptSource\..\Logs"
If (!(Test-Path "$LogFolder"))
{
	mkdir "$LogFolder" >$null
	mkdir "$LogFolder\Expand" >$null
}

$LogFile = "$LogFolder\Expand\$logFileName"
Start-Transcript -LiteralPath "$LogFile-Transcript.txt" -Append

Set-Location "$ScriptSource\..\Diskpart"
Logline "======================================================"
Logline "Expanding User Layer [$PathToVHD] to [$NewSize GB]"
Logline "======================================================"

[int]$SizeOfDiskParam = $NewSize
$SizeOfDisk = $SizeOfDiskParam * 1024

$disk = "$PathToVHD"

#Lets get the current Size 
NEW-ITEM name SizevDisk.TXT itemtype file force | OUT-NULL
ADD-CONTENT path SizevDisk.TXT "Select vdisk FILE=`'$disk`'" 
ADD-CONTENT path SizevDisk.TXT "Detail vDisk" 
ADD-CONTENT path SizevDisk.TXT "exit"
if (Test-Path SizevDisk.TXT)
{
	$DetailVDISK=(DISKPART /S SizevDisk.TXT)
	if ($DetailVDISK[6] -eq "DiskPart has encountered an error: The process cannot access the file because it is being used by another process.")
	{
		$PathToVHD | Out-File -LiteralPath DiskLocked.txt -Encoding ASCII
		Logline "User Layer [$PathToVHD] Is in use and cannot be modified at this time."
		exit
	}
}
Else
{
	Logline "****Error Creating Diskpart script files - Exiting Script"
	Exit
}

$SizeData = $DetailVDISK
foreach ($line in $SizeData)
{
	$arrLine = $line -split '\s+'
	if ([string]::IsNullorEmpty($arrLine[1])){continue}
	if ($arrLine[0] -eq "Virtual")
	{
		$CurrentDiskSize = $arrLine[2]
		Logline "Current size for disk $disk is $CurrentDiskSize" -ForegroundColor Black
	}
}

if ([int]$CurrentDiskSize -ge [int]$NewSize)
{
	Logline "***Error - New Size [$NewSize] is less than or equal to current size [$CurrentDiskSize] - Exiting Script"
	$CurrentDiskSize | Out-File -LiteralPath CurrentDiskSize.txt -Encoding ASCII
	"Skipped" | Out-File -LiteralPath NewDiskSize.txt -Encoding ASCII
	Logline "Detaching vDisk $disk"
	#Detach the vDisk
	NEW-ITEM name DetachvDisk.TXT itemtype file force | OUT-NULL
	ADD-CONTENT path DetachvDisk.TXT "Select vdisk FILE=`'$disk`'"
	ADD-CONTENT path DetachvDisk.TXT "detach vDisk noerr"
	ADD-CONTENT path DetachvDisk.TXT "exit"
	$DetachVDISK=(DISKPART /S DetachvDisk.TXT) 
	#Clean up
	Logline "Deleting DetachvDisk.TXT and SizevDisk.TXT"
	Del DetachvDisk.TXT
	Del SizevDisk.TXT
	Logline "Result = Skipped"
	sleep 2
	exit
}

$CurrentDiskSize | Out-File -LiteralPath CurrentDiskSize.txt -Encoding ASCII

#Lets first expand the vdisk to the calculated size
Logline "Expanding Disk [$disk]"
NEW-ITEM name ExpandDisk.TXT itemtype file force | OUT-NULL
ADD-CONTENT path ExpandDisk.TXT "Select vdisk FILE=`'$disk`'"
ADD-CONTENT path ExpandDisk.TXT "expand vDisk maximum=$SizeOfDisk noerr"
ADD-CONTENT path ExpandDisk.TXT "exit"
$EXPANDISK = (DISKPART /S ExpandDisk.TXT) 

Logline "Getting Disk Volume"
#Now we need to get the volume of our vDisk
#We have to assume its the last on in the list
NEW-ITEM name LISTVolume.TXT itemtype file force | OUT-NULL
ADD-CONTENT path LISTVolume.TXT "Select vdisk FILE=`'$disk`'"
ADD-CONTENT path LISTVolume.TXT "Attach vdisk"
ADD-CONTENT path LISTVolume.TXT "LIST VOLUME"
ADD-CONTENT path LISTVolume.TXT "exit"
$LISTVOL=(DISKPART /S LISTVolume.TXT) 

$VolumeData = $LISTVOL[-1]
foreach ($line in $LISTVOL)
{
	$arrLine = $line -split '\s+'
	if ([string]::IsNullorEmpty($arrLine[1])){continue}
	if ($arrLine[1] -eq "Volume")
	{
		$volNumber = $arrLine[2]
		$lastVolume = $volNumber
	}
}

Logline "The volume added is [$lastVolume]"

Logline "Extending the disk size for volume [$lastVolume]"
#Now we can extend the volume to its max size
NEW-ITEM name ExtendVolume.TXT itemtype file force | OUT-NULL
ADD-CONTENT path ExtendVolume.TXT "Select vdisk FILE=`'$disk`'"
ADD-CONTENT path ExtendVolume.TXT "Select Volume $lastVolume"
ADD-CONTENT path ExtendVolume.TXT "extend noerr"
ADD-CONTENT path ExtendVolume.TXT "exit"
$EXTENDVOL=(DISKPART /S ExtendVolume.TXT) 

Logline "Getting new disk size"
#Lets see how big it is after the extend
NEW-ITEM name SizevDisk.TXT itemtype file force | OUT-NULL
ADD-CONTENT path SizevDisk.TXT "Select vdisk FILE=`'$disk`'"
ADD-CONTENT path SizevDisk.TXT "Detail vDisk"
ADD-CONTENT path SizevDisk.TXT "exit"
$DetailVDISK=(DISKPART /S SizevDisk.TXT) 
$SizeData = $DetailVDISK
foreach ($line in $SizeData)
{
	#Write-Host "$line"
	$arrLine = $line -split '\s+'
	if ([string]::IsNullorEmpty($arrLine[1])){continue}
	if ($arrLine[0] -eq "Virtual")
	{
		$NewDiskSize = $arrLine[2]
		Logline "New size for disk $disk is $NewDiskSize" -ForegroundColor Black
	}
}

if ($NewDiskSize -eq $SizeOfDiskParam)
{
	Logline "Disk [$disk] was Successfully updated to $NewDiskSize GB"
	$result="Success"
}
else
{
	Logline "**** Warning - Disk [$disk] was NOT Successfully updated to $NewDiskSize GB"
	$result="Failure"
}
Logline "Detaching vDisk $disk"
NEW-ITEM name DetachvDisk.TXT itemtype file force | OUT-NULL
ADD-CONTENT path DetachvDisk.TXT "Select vdisk FILE=`'$disk`'"
ADD-CONTENT path DetachvDisk.TXT "detach vDisk noerr"
ADD-CONTENT path DetachvDisk.TXT "exit"
$DetachVDISK=(DISKPART /S DetachvDisk.TXT) 

$NewDiskSize | Out-File NewDiskSize.txt -Encoding ASCII

Logline "Cleaning Up Text Files DetachvDisk.TXT SizevDisk.TXT ExtendVolume.TXT ListVolume.TXT ExpandDisk.TXT"
Del DetachvDisk.TXT
Del SizevDisk.TXT
Del ExtendVolume.TXT
Del ListVolume.TXT
Del ExpandDisk.TXT
Logline "Script complete for User Layer [$disk]"
Logline "Result:$result"
Sleep 2
