
#Script CompressULAll.ps1
#Developed by Rob Zylowski Solution Delivery Architect Citrix Consulting Services.
#version 1.0 3-10-2021 - This script is intended to be used with the Citrix User Layer Repair Utility.  It will compress ALL user layer vhd files defined in the 
#user layer share.
#Version 2.0 5-23-2021 - Updated to add FSLogix 
#Version 3.0 7-28-2021 - UPdated to handle spaces in the disk path


<# *****************************************   LEGAL DISCLAIMER   *****************************************
This software / sample code is provided to you AS IS with no representations, warranties or conditions of any kind. 
You may use, modify and distribute it at your own risk. CITRIX DISCLAIMS ALL WARRANTIES WHATSOEVER, EXPRESS, IMPLIED, 
WRITTEN, ORAL OR STATUTORY, INCLUDING WITHOUT LIMITATION WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, 
TITLE AND NONINFRINGEMENT. Without limiting the generality of the foregoing, you acknowledge and agree that (a) the 
software / sample code may exhibit errors, design flaws or other problems, possibly resulting in loss of data or damage to 
property; (b) it may not be possible to make the software / sample code fully functional; and (c) Citrix may, without notice 
or liability to you, cease to make available the current version and/or any future versions of the software / sample code. 
In no event should the software / code be used to support of ultra-hazardous activities, including but not limited to life 
support or blasting activities. NEITHER CITRIX NOR ITS AFFILIATES OR AGENTS WILL BE LIABLE, UNDER BREACH OF CONTRACT OR ANY 
OTHER THEORY OF LIABILITY, FOR ANY DAMAGES WHATSOEVER ARISING FROM USE OF THE SOFTWARE / SAMPLE CODE, INCLUDING WITHOUT 
LIMITATION DIRECT, SPECIAL, INCIDENTAL, PUNITIVE, CONSEQUENTIAL OR OTHER DAMAGES, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH 
DAMAGES. Although the copyright in the software / code belongs to Citrix, any distribution of the code should include only 
your own standard copyright attribution, and not that of Citrix. You agree to indemnify and defend Citrix against any and all
claims arising from your use, modification or distribution of the code.
********************************************************************************************************** #>

function Get-ScriptDirectory
{
  $Invocation = (Get-Variable MyInvocation -Scope 1).Value
  Split-Path $Invocation.MyCommand.Path
}

Function LogLine($strLine)
{
	Write-Host $strLine
	$StrTime = Get-Date -Format "MM-dd-yyyy-HH-mm-ss-tt"
	"$StrTime - $strLine " | Out-file -FilePath $LogFile -Encoding ASCII -Append
}

$ScriptSource = Get-ScriptDirectory

$ProcessStartTime = Get-Date

$LogDateTime = Get-Date -Format "MM-dd-yyyy-HH-mm-tt"
$logFileName = "CompressAll-$LogDateTime-log.txt"
#Create a log folder and file
$LogFolder = "$ScriptSource\..\Logs"
If (!(Test-Path "$LogFolder"))
{
	mkdir "$LogFolder" >$null
	mkdir "$LogFolder\Compress" >$null
}
if (!(Test-Path "$LogFolder\Compress"))
{
	mkdir "$LogFolder\Compress" >$null
}
$LogFile = "$LogFolder\Compress\$logFileName"
Start-Transcript -LiteralPath "$LogFile-Transcript.txt" -Append

#Get the User Layer share

If (Test-Path "$ScriptSource\..\config\shares.txt")
{
	$Shares = Get-Content -Path "$ScriptSource\..\config\shares.txt" -Encoding ASCII
	$SharePath = $Shares[1]
	$UserSharePath = "$SharePath"
	$FSLogixProfileShare = $Shares[2]
	$FSLogixODFCShare = $Shares[3]

	Logline "User Share Set to [$SharePath]"
	Logline "FSLogix Profile Share [$FSLogixProfileShare]"
	Logline "FSLogix ODFC Share [$FSLogixODFCShare]"
}
else
{
	Logline "User Share settings not found.  Please run setup to create the shares.txt file in the config folder"
	Logline "Exiting Script"
	Logline "Result:Failure"
	Sleep 2
	exit
}

Set-Location "$ScriptSource\..\Diskpart"
Logline "======================================================"
Logline "Compressing User Layers in path [$UserSharePath]"
Logline "======================================================"

if ($UserSharePath -ne "" -and $UserSharePath -ne "\\server\share"){$UserLayerDisks = Get-ChildItem -include *.vhd,*vhdx -Path "$UserSharePath" -Recurse}
if ($FSLogixProfileShare -ne "" -and $FSLogixProfileShare -ne "\\server\share"){$FSLogixProfileDisks = Get-ChildItem -include *.vhd,*vhdx -Path "$FSLogixProfileShare" -Recurse}
if ($FSLogixODFCShare -ne "" -and $FSLogixProfileShare -ne "\\server\share"){$UserFSLogixODFCDisks = Get-ChildItem -include *.vhd,*vhdx -Path "$FSLogixODFCShare" -Recurse}
$AllDisks = $UserLayerDisks + $FSLogixProfileDisks + $UserFSLogixODFCDisks
foreach ($UserDisk in $AllDisks)
{

	$Disk  = $UserDisk.FullName
	Logline "===================================================="
	Logline "Processing user layer disk [$Disk]"
	Logline "===================================================="
	Logline ""
	#Lets get the current Sizes
	NEW-ITEM name SizevDisk.TXT itemtype file force | OUT-NULL
	ADD-CONTENT path SizevDisk.TXT "Select vdisk FILE=`'$disk`'" 
	ADD-CONTENT path SizevDisk.TXT "Detail vDisk" 
	ADD-CONTENT path SizevDisk.TXT "exit" 
	if (Test-Path SizevDisk.TXT)
	{
		$DetailVDISK=(DISKPART /S SizevDisk.TXT)
		if ($DetailVDISK[6] -eq "DiskPart has encountered an error: The process cannot access the file because it is being used by another process.")
		{
			Logline "User Layer [$Disk] Is in use and cannot be modified at this time."
			Logline "Cleaning Up Text Files DetachvDisk.TXT SizevDisk.TXT ExtendVolume.TXT ListVolume.TXT ExpandDisk.TXT"
			Del SizevDisk.TXT
			continue
		}
	}
	Else
	{
		Logline "****Error Creating Diskpart script files - Exiting Script"
		Exit
	}

	$SizeData = $DetailVDISK
	foreach ($line in $SizeData)
	{
		$arrLine = $line -split '\s+'
		if ([string]::IsNullorEmpty($arrLine[1])){continue}
		if ($arrLine[0] -eq "Virtual")
		{
			$CurrentDiskSize = $arrLine[2]
			Logline "Current size for disk $disk is $CurrentDiskSize" -ForegroundColor Black
		}
		if ($arrLine[0] -eq "Physical")
		{
			$PhysicalDiskSize = $arrLine[2]
			$PhysicalDiskUnits = $arrLine[3]
			Logline "Current space used within disk $disk is $PhysicalDiskSize $PhysicalDiskUnits" -ForegroundColor Black
		}
	}

	$PhysicalDiskBefore = $PhysicalDiskSize + ' ' + $PhysicalDiskUnits 
	
	$CompressSuccess=$False
	#Lets compact the disk
	Logline "Compressing Disk [$disk]"
	NEW-ITEM name CompressDisk.TXT itemtype file force | OUT-NULL
	ADD-CONTENT path CompressDisk.TXT "Select vdisk FILE=`'$disk`'"
	ADD-CONTENT path CompressDisk.TXT "attach vdisk readonly"
	ADD-CONTENT path CompressDisk.TXT "compact vdisk"
	ADD-CONTENT path CompressDisk.TXT "detach vdisk"
	ADD-CONTENT path CompressDisk.TXT "exit"
	$COMPRESSDISK = (DISKPART /S CompressDisk.TXT) 

	#Lets see if the compress worked
	If ($COMPRESSDISK -like '*DiskPart successfully compacted the virtual disk file*')
	{
		Logline "User Layer [$UserDisk] Compress was Successful"
		$CompressSuccess=$True
	}
	else
	{
		#$errortxt = $COMPRESSDISK[6]
		$errortxt = $COMPRESSDISK
		Logline "***User Layer [$UserDisk] Compress FAILED" 
		Logline "Error: $errortxt"
	}

	#If we are successful lets get the new size
	if ($CompressSuccess)
	{

		#Lets get the new  Sizes
		NEW-ITEM name SizevDisk.TXT itemtype file force | OUT-NULL
		ADD-CONTENT path SizevDisk.TXT "Select vdisk FILE=`'$disk`'" 
		ADD-CONTENT path SizevDisk.TXT "Detail vDisk" 
		ADD-CONTENT path SizevDisk.TXT "exit" 
		if (Test-Path SizevDisk.TXT)
		{
			$DetailVDISK=(DISKPART /S SizevDisk.TXT)
			if ($DetailVDISK[6] -eq "DiskPart has encountered an error: The process cannot access the file because it is being used by another process.")
			{
				Logline "**User Layer [$Disk] Is in use and the end size cannot be calculated"
				Del SizevDisk.TXT
				Del CompressDisk.TXT
				Logline "Skipping to next user layer disk"
				continue
			}
		}

		$SizeData = $DetailVDISK
		foreach ($line in $SizeData)
		{
			$arrLine = $line -split '\s+'
			if ([string]::IsNullorEmpty($arrLine[1])){continue}
			if ($arrLine[0] -eq "Physical")
			{
				$PhysicalDiskSizeAfter = $arrLine[2]
				$PhysicalDiskUnitsAfter = $arrLine[3]
				Logline "New space used within disk $disk is $PhysicalDiskSize $PhysicalDiskUnits" -ForegroundColor Black
			}
		}

		$PhysicalDiskAfter = $PhysicalDiskSizeAfter + ' ' + $PhysicalDiskUnitsAfter
		Logline "Size Before Compress : $PhysicalDiskBefore | Size After Compress: $PhysicalDiskAfter"
	}
	
	Logline "Cleaning Up Text Files DetachvDisk.TXT SizevDisk.TXT ExtendVolume.TXT ListVolume.TXT ExpandDisk.TXT"
	Del SizevDisk.TXT
	Del CompressDisk.TXT
	logline ""
}

sleep 10