﻿
#Script ConvertULAll.ps1
#Developed by Rob Zylowski Solution Delivery Architect Citrix Consulting Services.
#version 1.0 10-9-2023 - This script is intended to be used with the Citrix User Layer Repair Utility.  It will convert ALL user layer vhd files defined in the 
#user layer share to vhdx files if they are not in use runnin x number in parallel.


<# *****************************************   LEGAL DISCLAIMER   *****************************************
This software / sample code is provided to you “AS IS” with no representations, warranties or conditions of any kind. 
You may use, modify and distribute it at your own risk. CITRIX DISCLAIMS ALL WARRANTIES WHATSOEVER, EXPRESS, IMPLIED, 
WRITTEN, ORAL OR STATUTORY, INCLUDING WITHOUT LIMITATION WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, 
TITLE AND NONINFRINGEMENT. Without limiting the generality of the foregoing, you acknowledge and agree that (a) the 
software / sample code may exhibit errors, design flaws or other problems, possibly resulting in loss of data or damage to 
property; (b) it may not be possible to make the software / sample code fully functional; and (c) Citrix may, without notice 
or liability to you, cease to make available the current version and/or any future versions of the software / sample code. 
In no event should the software / code be used to support of ultra-hazardous activities, including but not limited to life 
support or blasting activities. NEITHER CITRIX NOR ITS AFFILIATES OR AGENTS WILL BE LIABLE, UNDER BREACH OF CONTRACT OR ANY 
OTHER THEORY OF LIABILITY, FOR ANY DAMAGES WHATSOEVER ARISING FROM USE OF THE SOFTWARE / SAMPLE CODE, INCLUDING WITHOUT 
LIMITATION DIRECT, SPECIAL, INCIDENTAL, PUNITIVE, CONSEQUENTIAL OR OTHER DAMAGES, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH 
DAMAGES. Although the copyright in the software / code belongs to Citrix, any distribution of the code should include only 
your own standard copyright attribution, and not that of Citrix. You agree to indemnify and defend Citrix against any and all
claims arising from your use, modification or distribution of the code.
********************************************************************************************************** #>

Param(
    [int]$SimultaneousConversions=4
)

function Get-ScriptDirectory
{
  $Invocation = (Get-Variable MyInvocation -Scope 1).Value
  Split-Path $Invocation.MyCommand.Path
}

Function LogLine($strLine)
{
	Write-Host $strLine
	$StrTime = Get-Date -Format "MM-dd-yyyy-HH-mm-ss-tt"
	"$StrTime - $strLine " | Out-file -FilePath $LogFile -Encoding ASCII -Append
}


function Test-FileLock {
  param (
    [parameter(Mandatory=$true)][string]$Path
  )

  $oFile = New-Object System.IO.FileInfo $Path

  if ((Test-Path -Path $Path) -eq $false) {
    return $false
  }

  try {
    $oStream = $oFile.Open([System.IO.FileMode]::Open, [System.IO.FileAccess]::ReadWrite, [System.IO.FileShare]::None)

    if ($oStream) {
      $oStream.Close()
    }
    return $false
  } catch {
    # file is locked by a process.
    return $true
  }
}

########### SETTINGS ###########
#Timelimit in Hours
$TimeLimit=10
################################

Import-Module Hyper-V

$ScriptSource = Get-ScriptDirectory

$ProcessStartTime = Get-Date

$LogDateTime = Get-Date -Format "MM-dd-yyyy-HH-mm-tt"
$logFileName = "ConvertAllJobs-$LogDateTime-log.txt"
#Create a log folder and file
$LogFolder = "$ScriptSource\..\Logs"
If (!(Test-Path "$LogFolder"))
{
	mkdir "$LogFolder" >$null
	mkdir "$LogFolder\Convert" >$null
}
if (!(Test-Path "$LogFolder\Convert"))
{
	mkdir "$LogFolder\Convert" >$null
}
$LogFile = "$LogFolder\Convert\$logFileName"
#Start-Transcript -LiteralPath "$LogFile-Transcript.txt" -Append

Logline "Number of Simulatneous Conversions is [$SimultaneousConversions]"

#Get the User Layer share

If (Test-Path "$ScriptSource\..\config\shares.txt")
{
	$Shares = Get-Content -Path "$ScriptSource\..\config\shares.txt" -Encoding ASCII
	$SharePath = $Shares[1]
	$UserSharePath = "$SharePath"

	Logline "User Share Set to [$SharePath]"
}
else
{
	Logline "User Share settings not found.  Please run setup to create the shares.txt file in the config folder"
	Logline "Exiting Script"
	Logline "Result:Failure"
	Sleep 2
	exit
}

Logline "======================================================"
Logline "Converting User Layers in path [$UserSharePath]"
Logline "======================================================"

if ($UserSharePath -ne "" -and $UserSharePath -ne "\\server\share")
{
    $AllDisks = Get-ChildItem -include *.vhd -Path "$UserSharePath" -Recurse

}

#Create and load a data table to manage the job status
$VHDTable = New-Object system.Data.DataTable "VHDTable"
[array]$Columns = $("VHDPath","EncryptedVHDPath","Status","ConvStatus","InstanceId")
ForEach ($Column in $Columns) {$VHDTable.Columns.Add($(New-Object system.Data.DataColumn $Column,([string])))}

#Import the input CSV into the data table
foreach ($UserDisk in $AllDisks)
{
	#Format of desktop name domain\computer
	$VHDPath = $UserDisk.FullName
    $encodedBytes = [System.Text.Encoding]::unicode.GetBytes($VHDPath)
    $VHDText = [System.Convert]::ToBase64String($encodedBytes)

	#Now lets create the row for this record in our table
	$TableRow = $VHDTable.NewRow() 
	$TableRow.VHDPath = $VHDPath
    $TableRow.EncryptedVHDPath = $VHDText
	$TableRow.Status = "Pending"
    $TableRow.ConvStatus = "Pending"
	$TableRow.InstanceId = "None"
	$VHDTable.Rows.Add($TableRow)	
}

Set-location $scriptsource

$ProcessStartTime = Get-Date
$LoopTimeLimit = $ProcessStartTime.AddHours($TimeLimit)

$pool = [RunspaceFactory]::CreateRunspacePool(1, $SimultaneousConversions)
$pool.ApartmentState = "MTA"
$pool.Open()
$runspaces = @()

#Number of desktops to process
$TotalVHDs = $VHDTable.Rows.Count
$count = 0

foreach ($Row in $VHDTable.Rows)
{
	$PathToVHD = $Row.VHDPath
    $PathToVHDEncrypted = $Row.EncryptedVHDPath
	
	if ($UseTimeLimit)
	{
		Logline "Time Limit in Use -- Checking Time"
		#Check for time limit
		#Exit if we past the time limit
		$CurDateTime = Get-Date
		$bTimeLimitReached = $false
		if ($CurDateTime -gt $LoopTimeLimit)
		{
			Logline "*****VHD [$PathToVHD] was skipped due to processing time limit"
            $row.ConvStatus = "SkippedTime"
			$row.Status = "Failure"
			continue
		}
	}
	
	if ($Row.Status -ne "Completed")
	{
		#Wait for a thread to become available.
		$AvailRunspaces =  $Pool.GetAvailableRunspaces()
		Write-Host "Available Runspaces = [$AvailRunspaces]"
		#will wait for an available slot in the runspaces pool
		While ($AvailRunspaces -eq 0)
		{
			$AvailRunspaces =  $Pool.GetAvailableRunspaces()
			Write-Host "." -NoNewline
			Sleep 60
			continue
		}
		
		if ($Row.Status -ne "Started")
		{
			$count++
			$runspace = [PowerShell]::Create()
			$runspaceID=$runspace.InstanceId
			$script = "`'$ScriptSource\ConvertUL.ps1`'"
			#$null = 
            $runspace.AddScript("& $script $PathToVHDEncrypted")
			$runspace.RunspacePool = $pool
			
			logline "Converting VHD [$PathToVHD] Processing [$count of $TotalVHDs]"  
		 	
			$runspaces += [PSCustomObject]@{ Pipe = $runspace; Status = $runspace.BeginInvoke() }
			$Row.Status = "Started"
			$Row.InstanceId = "$runspaceID"
		}
	}
}

logline "Waiting for last jobs to finish"
$AvailRunspaces =  $Pool.GetAvailableRunspaces()
while ($AvailRunspaces -lt $SimultaneousJobs)
{
	$Jobsremaining = $SimultaneousJobs - $AvailRunspaces
	Write-Host "[$Jobsremaining] Active Jobs Running" 
	Sleep 60
	$AvailRunspaces =  $Pool.GetAvailableRunspaces()
}

#Convert Flags
$Success = 0
$Failure = 0
$SkippedTime = 0
$SkippedInUse = 0 
$SkippedExists = 0
$arrSuccess = @()
$arrFailure = @()
$arrSkippedTime = @()
$arrSkippedInUse = @() 
$arrSkippedExists = @()

foreach ($row in $VHDTable) 
{
    if (!($row.ConvStatus -eq "SkipTime"))
    {
        $RunSpaceId=$row.InstanceId
        $job = get-runspace -InstanceId "$RunSpaceId"
        foreach ($job in $Runspaces)
        {
            if ($job.InstanceID -eq $RunSpaceId)
            {
                break
            }
        }

        $PathToVHD=$row.VHDPath
	    $jobReturnAll = $job.Pipe.EndInvoke($job.status)
        if ($jobReturnAll.Count -eq 1)
        {
             $jobReturn = $jobReturnAll
        }
        else
        {
             $jobReturn = $jobReturnAll[1]
        }
       

	    Switch($jobReturn) 
	    {
		    "Success" {$success++;$arrSuccess+="$PathToVHD`r`n"}
		    "Failure" {$Failure++;$arrFailure+="$PathToVHD`r`n"}
		    "SkippedTime" {$SkippedTime++;$arrSkippedTime+="$PathToVHD`r`n"} 
		    "SkippedinUse" {$SkippedInUse++;$arrSkippedInUse+="$PathToVHD`r`n"}
            "SkippedExists" {$SkippedExists++;$arrSkippedExists+="$PathToVHD`r`n"}
	    }

	    #Check the runspace for this row and if the job IsCompleted

		$HadErrors = $job.Pipe.HadErrors
		$ReturnedData = $job.Pipe.EndInvoke($job.status)

		if ($HadErrors)
		{
			Logline "==========================="
			Logline "*** Error for VHD [$PathToVHD]-"
			Logline "*** [$ReturnedData]"
			Logline "Errors"
			Logline "----------------------------"
			$ErrorText = $job.Pipe.Streams.Error
			Logline "$ErrorText"
			Logline "==========================="
		}
		else
		{
			Logline ""
			Logline "VHD [$PathToVHD] has finished processing without errors"
		}
		
		$job.Pipe.Dispose()
		$Row.Status = "Completed"
    }
    else
    {
        $SkippedTime++;$arrSkippedTime+="$PathToVHD`r`n"
    }
}

	

Logline "Cleanup jobs from memory"
$pool.Close() 
$pool.Dispose()

Logline ""
Logline ""
Logline "Batch Statistics"
Logline "================================="

Logline "[$success] VHDs Converted"
foreach ($line in $arrSuccess)
{
	Logline "$line"
}
Logline "[$Failure] VHD NOT Converted"
foreach ($line in $arrFailure)
{
    Logline "$line"
}
Logline "[$SkippedTime] VHDs Skipped Due to Time Limit"
foreach ($line in $arrSkippedTime)
{
	Logline "$line"
}
Logline "[$SkippedInUse] VHDs Skipped In Use"
foreach ($line in $arrSkippedInUse)
{
	Logline "$line"
}
Logline "[$SkippedExists] VHDs skipped already migrated"
foreach ($line in $arrSkippedExists)
{
	Logline "$line"
}

$ProcessEndTime = Get-date

$timeinterval = New-TimeSpan -Start $ProcessStartTime -End $ProcessEndTime
$Hours = $timeinterval.Hours
$minutes = $timeinterval.Minutes

Logline "================================="
Logline "Processing Time [$Hours] Hours [$minutes] Minutes"
Logline "================================="

#stop-transcript