
#Script ConvertUL.ps1
#Developed by Rob Zylowski Solution Delivery Architect Citrix Consulting Services.
#Version 1.0 10-7-2023 - This script is intended to be used with the Citrix User Layer Repair Utility.  It will convert selected user layer vhd files to vhdx using the HyperV module.

<# *****************************************   LEGAL DISCLAIMER   *****************************************
This software / sample code is provided to you AS IS with no representations, warranties or conditions of any kind. 
You may use, modify and distribute it at your own risk. CITRIX DISCLAIMS ALL WARRANTIES WHATSOEVER, EXPRESS, IMPLIED, 
WRITTEN, ORAL OR STATUTORY, INCLUDING WITHOUT LIMITATION WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, 
TITLE AND NONINFRINGEMENT. Without limiting the generality of the foregoing, you acknowledge and agree that (a) the 
software / sample code may exhibit errors, design flaws or other problems, possibly resulting in loss of data or damage to 
property; (b) it may not be possible to make the software / sample code fully functional; and (c) Citrix may, without notice 
or liability to you, cease to make available the current version and/or any future versions of the software / sample code. 
In no event should the software / code be used to support of ultra-hazardous activities, including but not limited to life 
support or blasting activities. NEITHER CITRIX NOR ITS AFFILIATES OR AGENTS WILL BE LIABLE, UNDER BREACH OF CONTRACT OR ANY 
OTHER THEORY OF LIABILITY, FOR ANY DAMAGES WHATSOEVER ARISING FROM USE OF THE SOFTWARE / SAMPLE CODE, INCLUDING WITHOUT 
LIMITATION DIRECT, SPECIAL, INCIDENTAL, PUNITIVE, CONSEQUENTIAL OR OTHER DAMAGES, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH 
DAMAGES. Although the copyright in the software / code belongs to Citrix, any distribution of the code should include only 
your own standard copyright attribution, and not that of Citrix. You agree to indemnify and defend Citrix against any and all
claims arising from your use, modification or distribution of the code.
********************************************************************************************************** #>

Param(
[parameter(Mandatory=$true)]
$EncodedPathToVHD,

[switch]$DelAfterConvert,

[boolean]$CompressBeforeConvert=$true,

$VHDXSubformat="dynamic"

)

function Get-ScriptDirectory
{
  $Invocation = (Get-Variable MyInvocation -Scope 1).Value
  Split-Path $Invocation.MyCommand.Path
}

Function LogLine($strLine)
{
	Write-Host $strLine
	$StrTime = Get-Date -Format "MM-dd-yyyy-HH-mm-ss-tt"
	"$StrTime - $strLine " | Out-file -FilePath $LogFile -Encoding ASCII -Append
}

function Test-FileLock {
  param (
    [parameter(Mandatory=$true)][string]$Path
  )

  $oFile = New-Object System.IO.FileInfo $Path

  if ((Test-Path -Path $Path) -eq $false) {
    return $false
  }

  try {
    $oStream = $oFile.Open([System.IO.FileMode]::Open, [System.IO.FileAccess]::ReadWrite, [System.IO.FileShare]::None)

    if ($oStream) {
      $oStream.Close()
    }
    return $false
  } catch {
    # file is locked by a process.
    return $true
  }
}

Import-module Hyper-V

$ScriptSource = Get-ScriptDirectory

$ProcessStartTime = Get-Date

#Create a log folder and file
$LogFolder = "$ScriptSource\..\Logs"
If (!(Test-Path "$LogFolder"))
{
	mkdir "$LogFolder" >$null
	mkdir "$LogFolder\Convert" >$null
    mkdir "$LogFolder\Convert\VHDS" >$null
}

If (!(Test-Path "$LogFolder\Convert"))
{
	mkdir "$LogFolder\Convert" >$null
}

If (!(Test-Path "$LogFolder\Convert\VHDS"))
{
    mkdir "$LogFolder\Convert\VHDS" >$null
}

#We are using a base64 string to pass the vhd filename to handle spaces in the path.
#This will convert it back for us.
$PathToVHD=[System.Text.Encoding]::Unicode.GetString([System.Convert]::FromBase64String($EncodedPathToVHD))
$PathToVHDTrans = $PathToVHD.Substring(2,$PathToVHD.Length-6) 

$logFileName = $PathToVHDTrans -replace "\\","-"
$LogFile = "$LogFolder\Convert\VHDS\$logFileName.txt"

#Start-Transcript -LiteralPath "$LogFile-Transcript.txt" -Append

Logline "======================================================"
Logline "Converting User Layer [$PathToVHD] from VHD to VHDX"
Logline "======================================================"

#Delete the success flag file if its there
if (Test-FileLock "$ScriptSource\..\config\convertsuccess.txt")
{
    del "$ScriptSource\..\config\convertsuccess.txt"
}

#First Lets make sure the vhd file is not locked
if (Test-FileLock $PathToVHD)
{
    Logline "VHD file [$PathToVHD] is locked and cannot be processed"
    #Do somethig to report back that it is locked.
    "SkippedInUse" | out-file "$ScriptSource\..\config\convertstatus.txt" -Encoding ASCII
    $ReturnStatus = "SkippedInUse"
    $ReturnStatus
    Exit
}
else
{
    Logline "VHD file [$PathToVHD] is not locked and can be processed"
}

$PathToVHDX = "$PathToVHD" + "x"

Logline "Path to VHDX [$PathToVHDX]"

#First Lets make sure the VHDX does not already exist
if (Test-Path $PathToVHDX)
{
    Logline "!!! Conversion Skipped !!! - The converted VHDX file was found before the conversions."
    "SkippedExists" | out-file "$ScriptSource\..\config\convertstatus.txt" -Encoding ASCII
    $ReturnStatus = "SkippedExists"
    $ReturnStatus
    exit
}

if ($CompressBeforeConvert)
{
    #First Compress the vhd before converting
    Logline "Compressing VHD [$PathToVHD]"
    Mount-VHD -Path "$PathToVHD" -ReadOnly 
    Optimize-VHD -Path "$PathToVHD" -Mode Full
    DISMOUNT-VHD -Path "$PathToVHD"
}

#Now lets convert the vhd to vhdx
Logline "Running Convert-vhd"

if($DelAfterConvert)
{
    Logline "Will DELETE after successful conversion"
    Convert-vhd -path "$PathToVHD" -destinationpath "$PathToVHDX" -VHDType $VHDXSubformat -DeleteSource
}
else
{
    Logline "Will NOT DELETE after successful conversion"
    Convert-vhd -path "$PathToVHD" -destinationpath "$PathToVHDX" -VHDType $VHDXSubformat
}

if (!$?)
{
    Logline "There ws an error with the conversion"
    Logline $error[0]
}


#Now find the vhdx make sure its there
if (Test-Path "$PathToVHDX")
{
	Logline "!!! Conversion Success !!! - The converted VHDX file was found in the share."
    "Success" | out-file "$ScriptSource\..\config\convertstatus.txt" -Encoding ASCII
    $ReturnStatus = "Success" 
}
else
{
	Logline "*** Conversion Failure *** - The VHDX file was NOT found in the share."
    "Failure" | out-file "$ScriptSource\..\config\convertstatus.txt" -Encoding ASCII
    $ReturnStatus = "Failure"
}

Sleep 5
$ReturnStatus